<?php
namespace App\Notifications;

use App\Channels\SmsMessage;
use App\Models\Comment;
use App\Models\EmailTemplate;
use App\Utilities\Overrider;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TaskCommented extends Notification implements ShouldQueue {
    use Queueable;

    public $comment;
    private $template;
    private $replace = [];

    /**
     * Create a new notification instance.
     *
     * @param \App\Models\Comment $comment
     */
    public function __construct(Comment $comment) {
        Overrider::load("Settings");
        $this->comment = $comment;

        $this->template = EmailTemplate::where('slug', 'TASK_COMMENTED')
            ->where('tenant_id', request()->tenant->id)
            ->first();

        $this->replace['commenter'] = $this->comment->user->name;
        $this->replace['title']     = $this->comment->task->title;
        $this->replace['taskUrl']   = route('tasks.show', $this->comment->task->id);
        $this->replace['dueDate']   = $this->comment->task->due_date;
    }

    public function via($notifiable) {
        $channels = [];
        if ($this->template != null && $this->template->email_status == 1) {
            array_push($channels, 'mail');
        }
        if ($this->template != null && $this->template->sms_status == 1) {
            array_push($channels, \App\Channels\SMS::class);
        }
        if ($this->template != null && $this->template->notification_status == 1) {
            array_push($channels, 'database');
        }
        return $channels;
    }

    public function toMail($notifiable) {
        $message = processShortCode($this->template->email_body, $this->replace);
        return (new MailMessage)
            ->subject($this->template->subject)
            ->markdown('email.notification', ['message' => $message]);
    }

    /**
     * Get the sms representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toSMS($notifiable) {
        $message = processShortCode($this->template->sms_body, $this->replace);

        return (new SmsMessage())
            ->setContent($message)
            ->setRecipient($notifiable->phone);
    }

    public function toDatabase($notifiable) {
        $message = processShortCode($this->template->email_body, $this->replace);
        return [
            'subject' => $this->template->subject,
            'message' => $message,
        ];
    }
}
