<?php
namespace App\Notifications;

use App\Channels\SmsMessage;
use App\Models\EmailTemplate;
use App\Models\Task;
use App\Utilities\Overrider;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TaskAssigned extends Notification {
    use Queueable;

    private $task;
    private $template;
    private $replace = [];

    public function __construct(Task $task, $employeeName) {
        Overrider::load("Settings");
        $this->task = $task;

        $this->template = EmailTemplate::where('slug', 'TASK_ASSIGNED')
            ->where('tenant_id', request()->tenant->id)
            ->first();

        $this->replace['employeeName'] = $employeeName;
        $this->replace['title']        = $this->task->title;
        $this->replace['taskUrl']      = route('tasks.show', $this->task->id);
        $this->replace['dueDate']      = $this->task->due_date;
    }

    public function via($notifiable) {
        $channels = [];
        if ($this->template != null && $this->template->email_status == 1) {
            array_push($channels, 'mail');
        }
        if ($this->template != null && $this->template->sms_status == 1) {
            array_push($channels, \App\Channels\SMS::class);
        }
        if ($this->template != null && $this->template->notification_status == 1) {
            array_push($channels, 'database');
        }
        return $channels;
    }

    public function toMail($notifiable) {
        $message = processShortCode($this->template->email_body, $this->replace);
        return (new MailMessage)
            ->subject($this->template->subject)
            ->markdown('email.notification', ['message' => $message]);
    }

    /**
     * Get the sms representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toSMS($notifiable) {
        $message = processShortCode($this->template->sms_body, $this->replace);

        return (new SmsMessage())
            ->setContent($message)
            ->setRecipient($notifiable->phone);
    }

    public function toDatabase($notifiable) {
        $message = processShortCode($this->template->email_body, $this->replace);
        return [
            'subject' => $this->template->subject,
            'message' => $message,
        ];
    }
}
