<?php
namespace App\Models;

use App\Observers\TaskObserver;
use App\Traits\MultiTenant;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;

#[ObservedBy([TaskObserver::class])]
class Task extends Model {
    use MultiTenant, HasFactory;

    protected $fillable = ['title', 'description', 'due_date', 'user_id', 'status', 'custom_fields'];

    // Cast custom_fields to array automatically
    protected $casts = [
        'custom_fields' => 'array',
    ];

    // Relationships
    public function user() {
        return $this->belongsTo(User::class);
    }

    public function comments() {
        return $this->hasMany(Comment::class);
    }

    public function files() {
        return $this->hasMany(File::class);
    }

    public function subtasks() {
        return $this->hasMany(Subtask::class);
    }

    // Many-to-many: users assigned to this task
    public function users() {
        return $this->belongsToMany(User::class);
    }

    // Task dependencies: tasks that this task depends on
    public function dependencies() {
        return $this->belongsToMany(Task::class, 'task_dependencies', 'task_id', 'dependency_id');
    }

    // Tasks that depend on this task
    public function dependents() {
        return $this->belongsToMany(Task::class, 'task_dependencies', 'dependency_id', 'task_id');
    }

    public function logs() {
        return $this->hasMany(TaskLog::class);
    }

    public function getCustomFieldsAttribute($value) {
        return json_decode($value, false);
    }

    protected function dueDate(): Attribute {
        $date_format = get_date_format();

        return Attribute::make(
            get: fn($value) => \Carbon\Carbon::parse($value)->format("$date_format"),
        );
    }
}
