<?php
namespace App\Http\Controllers;

use App\Models\Subtask;
use App\Models\Task;
use App\Models\TaskLog;
use App\Models\User;
use App\Notifications\TaskAssigned;
use App\Notifications\TaskStatusUpdated;
use DataTables;
use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class TaskController extends Controller {

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct() {
        date_default_timezone_set(get_timezone());

        $this->middleware(function ($request, $next) {
            if ($request->tenant->package->task_management != 1) {
                if ($request->ajax()) {
                    return response()->json(['result' => 'error', 'message' => _lang('Sorry, This module is not available in your subscription plan !')]);
                }
                return back()->with('error', _lang('Sorry, This module is not available in your subscription plan !'));
            }

            return $next($request);
        });
    }

    public function index() {
        $assets = ['datatable'];
        return view('tasks.index', compact('assets'));
    }

    public function get_table_data() {
        $tasks = Task::with('users', 'user')
            ->where('user_id', Auth::id())
            ->orWhereHas('users', function ($q) {
                $q->where('users.id', Auth::id());
            })->orderBy('due_date', 'asc');

        return Datatables::eloquent($tasks)
            ->editColumn('title', function ($task) {
                return Str::limit($task->title, 40);
            })
            ->editColumn('users', function ($task) {
                $users = '';
                foreach ($task->users as $user) {
                    $users .= $user->name . ', ';
                }
                return $users ? rtrim($users, ', ') : _lang('No users assigned');
            })
            ->editColumn('status', function ($task) {
                return '<div class="text-center">' . task_status($task->status) . '</div>';
            })
            ->addColumn('action', function ($task) {
                return '<div class="dropdown text-center">'
                . '<button class="btn btn-outline-primary btn-xs dropdown-toggle" type="button" data-toggle="dropdown">' . _lang('Action')
                . '</button>'
                . '<div class="dropdown-menu">'
                . '<a class="dropdown-item" href="' . route('tasks.show', $task['id']) . '"><i class="fas fa-eye"></i> ' . _lang('Task Details') . '</a>'
                . '<a class="dropdown-item" href="' . route('tasks.edit', $task['id']) . '"><i class="fas fa-pencil-alt"></i> ' . _lang('Edit Task') . '</a>'
                . '<form action="' . route('tasks.destroy', $task['id']) . '" method="post">'
                . csrf_field()
                . '<input name="_method" type="hidden" value="DELETE">'
                . '<button class="dropdown-item btn-remove" type="submit"><i class="fas fa-trash-alt"></i> ' . _lang('Delete') . '</button>'
                    . '</form>'
                    . '</div>'
                    . '</div>';
            })
            ->setRowId(function ($notice) {
                return "row_" . $notice->id;
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    public function kanban() {
        $tasks = Task::where('user_id', Auth::id())
            ->orWhereHas('users', function ($q) {
                $q->where('users.id', Auth::id());
            })
            ->orderBy('due_date')
            ->get()
            ->groupBy('status');
        $taskColors = ['To_Do' => '#dc3545', 'In_Progress' => '#4834d4', 'Completed' => '#27ae60'];
        return view('tasks.kanban', compact('tasks', 'taskColors'));
    }

    public function create() {
        $alert_col = 'col-lg-8 offset-lg-2';
        $assets    = ['tinymce'];
        $allUsers  = User::all();
        $allTasks  = Task::all();
        return view('tasks.create', compact('allUsers', 'allTasks', 'alert_col', 'assets'));
    }

    public function store(Request $request) {
        $request->validate([
            'title'          => 'required|string|max:191',
            'due_date'       => 'nullable|date',
            'description'    => 'required|string',
            'custom_fields'  => 'nullable|array',
            'user_ids'       => 'nullable|array',
            'dependency_ids' => 'nullable|array',
        ]);

        $customFields = $request->input('custom_fields', []);
        $customData   = [];

        if ($customFields) {
            foreach ($customFields as $field) {
                // Only store fields with a non-empty key
                if (isset($field['key']) && ! empty($field['key'])) {
                    $customData[$field['key']] = $field['value'] ?? '';
                }
            }
        }

        $task = Task::create([
            'title'         => $request->title,
            'due_date'      => $request->due_date,
            'description'   => $request->description,
            'user_id'       => Auth::id(),
            'custom_fields' => $customData,
        ]);

        if ($request->has('user_ids')) {
            $task->users()->sync($request->user_ids);

            $users = User::whereIn('id', $request->user_ids)->pluck('name')->toArray();
            TaskLog::create([
                'task_id'     => $task->id,
                'user_id'     => Auth::id(),
                'action'      => 'users_assigned',
                'description' => "Task assigned to users: " . implode(', ', $users),
                'data'        => ['assigned_users' => $users],
            ]);

            // Notify each assigned user about the new task assignment.
            foreach ($request->user_ids as $userId) {
                $user = User::find($userId);
                if ($user) {
                    try {
                        $user->notify(new TaskAssigned($task, $user->name));
                    } catch (Exception $e) {
                        //Nothing
                    }
                }
            }
        }

        if ($request->has('dependency_ids')) {
            $task->dependencies()->sync($request->dependency_ids);

            TaskLog::create([
                'task_id'     => $task->id,
                'user_id'     => Auth::id(),
                'action'      => 'dependencies_added',
                'description' => "Dependencies added to task.",
                'data'        => ['dependency_ids' => $request->dependency_ids],
            ]);
        }

        return redirect()->route('tasks.show', $task->id)->with('success', _lang('Task created successfully'));
    }

    public function show($tenant, $id) {
        $alert_col = 'col-lg-10 offset-lg-1';
        $assets    = ['tinymce'];
        $task      = Task::with('comments.user', 'files', 'subtasks', 'users', 'dependencies', 'dependents')->findOrFail($id);
        $allUsers  = User::all();
        $allTasks  = Task::where('id', '!=', $id)->get(); // Exclude current task from dependency list
        return view('tasks.show', compact('task', 'allUsers', 'allTasks', 'alert_col', 'assets'));
    }

    public function edit($tenant, $id) {
        $alert_col = 'col-lg-8 offset-lg-2';
        $assets    = ['tinymce'];
        $task      = Task::where('user_id', Auth::id())
            ->orWhereHas('users', function ($q) {
                $q->where('users.id', Auth::id());
            })->findOrFail($id);
        $allUsers = User::all();
        $allTasks = Task::where('id', '!=', $id)->get(); // Exclude self from dependency list
        return view('tasks.edit', compact('task', 'allUsers', 'allTasks', 'alert_col', 'assets'));
    }

    public function update(Request $request, $tenant, $id) {
        $request->validate([
            'title'          => 'required|string|max:191',
            'due_date'       => 'nullable|date',
            'description'    => 'required|string',
            'custom_fields'  => 'nullable|array',
            'dependency_ids' => 'nullable|array',
        ]);

        $customFields = $request->input('custom_fields', []);
        $customData   = [];

        if ($customFields) {
            foreach ($customFields as $field) {
                // Only store fields with a non-empty key
                if (isset($field['key']) && ! empty($field['key'])) {
                    $customData[$field['key']] = $field['value'] ?? '';
                }
            }
        }

        $task = Task::findOrFail($id);
        try {
            $this->authorize('update', $task);
        } catch (AuthorizationException $e) {
            return redirect()->back()->with('error', _lang('You are not authorized to update this task'));
        }

        // Capture old dependency assignments before syncing.
        $oldDependencyIds = $task->dependencies()->pluck('tasks.id')->toArray();

        $task->update([
            'title'         => $request->title,
            'due_date'      => $request->due_date,
            'description'   => $request->description,
            'custom_fields' => $customData,
        ]);

        $task->dependencies()->sync($request->dependency_ids ?? []);

        // Fetch the new assignments from the database in case they were modified by sync.
        $currentDependencyIds = $task->dependencies()->pluck('tasks.id')->toArray();

        // Log changes if dependencies have changed.
        if ($oldDependencyIds != $currentDependencyIds) {
            $dependencies = Task::whereIn('id', $currentDependencyIds)->pluck('title')->toArray();
            TaskLog::create([
                'task_id'     => $task->id,
                'user_id'     => Auth::id(),
                'action'      => 'dependencies_updated',
                'description' => count($dependencies) > 0 ? 'Task dependencies changed' : 'Task dependencies removed',
                'data'        => ['task_dependency' => $dependencies],
            ]);
        }

        return redirect()->route('tasks.show', $task->id)->with('success', _lang('Task updated successfully'));
    }

    public function update_status(Request $request, $tenant, $id) {
        $task = Task::with('dependencies')->findOrFail($id);

        $this->authorize('updateStatus', $task);

        if ($request->status === 'Completed') {
            foreach ($task->dependencies as $dependency) {
                if ($dependency->status !== 'Completed') {
                    return response()->json(['error' => _lang('Dependencies are not completed !')], 422);
                }
            }
        }

        $task->update(['status' => $request->status]);

        try {
            $task->user->notify(new TaskStatusUpdated($task));
        } catch (Exception $e) {
            //Nothing
        }

        return response()->json(['success' => _lang('Task status updated')]);
    }

    public function update_sub_task_status(Request $request, $tenant, $subtaskId) {
        $subtask = Subtask::findOrFail($subtaskId);

        $this->authorize('updateSubTaskStatus', $subtask->task);

        $oldStatus = str_replace('_', ' ', $subtask->getOriginal('status'));
        $subtask->update(['status' => $request->status]);
        $newStatus = str_replace('_', ' ', $subtask->status);

        TaskLog::create([
            'task_id'     => $subtask->task->id,
            'user_id'     => Auth::id(),
            'action'      => 'subtask_status_changed',
            'description' => "Sub Task status changed from {$oldStatus} to {$newStatus}",
            'data'        => $subtask->getChanges(),
        ]);

        return response()->json(['success' => _lang('Sub Task status updated')]);
    }

    public function destroy($tenant, $id) {
        $task = Task::findOrFail($id);
        try {
            $this->authorize('delete', $task);
        } catch (AuthorizationException $e) {
            return redirect()->back()->with('error', _lang('You are not authorized to delete this task'));
        }
        $task->delete();
        return redirect()->route('tasks.index')->with('success', _lang('Task deleted successfully'));
    }

    public function assign_users(Request $request, $tenant, $taskId) {
        $request->validate([
            'user_ids' => 'required|array',
        ]);

        $task            = Task::findOrFail($taskId);
        $existingUserIds = $task->users()->pluck('user_id')->toArray();
        $task->users()->sync($request->user_ids);

        $newUserIds = array_diff($request->user_ids, $existingUserIds);

        $users = User::whereIn('id', $request->user_ids)->pluck('name')->toArray();
        TaskLog::create([
            'task_id'     => $task->id,
            'user_id'     => Auth::id(),
            'action'      => 'users_assigned',
            'description' => count($users) > 0 ? 'User assigned to this task' : 'User removed from this task',
            'data'        => ['assigned_users' => $users],
        ]);

        // Notify each assigned user about the new assignment.
        foreach ($newUserIds as $userId) {
            $user = User::find($userId);
            if ($user) {
                try {
                    $user->notify(new TaskAssigned($task, $user->name));
                } catch (Exception $e) {
                    //Nothing
                }
            }
        }

        return back()->with('success', _lang('Users assigned to task successfully'));
    }

    // Add a subtask to a task
    public function add_sub_task(Request $request, $tenant, $taskId) {
        $request->validate([
            'title'       => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);
        $task = Task::findOrFail($taskId);
        $task->subtasks()->create([
            'title'       => $request->title,
            'description' => $request->description,
        ]);

        // Log comment posting action
        TaskLog::create([
            'task_id'     => $task->id,
            'user_id'     => Auth::id(),
            'action'      => 'subtask_added',
            'description' => "Sub Task created with title: {$request->title}",
        ]);

        return back()->with('success', _lang('Subtask added successfully'));
    }

    // Upload a file for a task
    public function upload_file(Request $request, $tenant, $taskId) {
        $request->validate([
            'file' => 'required|mimes:png,jpg,jpeg,pdf,doc,docx,xlsx,zip|max:4096',
        ]);
        $task = Task::findOrFail($taskId);
        $file = $request->file('file');
        $path = $file->store('task_files', 'public');

        $taskFile = $task->files()->create([
            'file_path' => $path,
            'file_name' => $file->getClientOriginalName(),
        ]);

        // Log file attachment action
        TaskLog::create([
            'task_id'     => $task->id,
            'user_id'     => Auth::id(),
            'action'      => 'file_attached',
            'description' => "File '{$taskFile->file_name}' attached to task.",
            'data'        => ['file_name' => $taskFile->file_name],
        ]);

        return back()->with('success', _lang('File uploaded successfully'));
    }
}
