<?php
namespace App\Http\Controllers\Employee;

use Carbon\Carbon;
use App\Models\Holiday;
use App\Models\WorkingHour;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class WorkingHourController extends Controller {

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct() {
        date_default_timezone_set(get_timezone());
    }

    public function clock_in(Request $request) {
        $employeeId = auth()->user()->employee->id;
        $today      = Carbon::today();

        $weekends = json_decode(get_tenant_option('weekends', '[]'));
        if (in_array(date('l', strtotime($today)), $weekends)) {
            return back()->with('error', _lang('Sorry, Today is weekend !'));
        }

        $holiday = Holiday::where('date', $today)->first();
        if ($holiday) {
            return back()->with('error', _lang('Sorry, Today is a holiday !'));
        }

        $clockInTime = Carbon::now()->format('H:i:s'); // Only store time

        // Create new clock-in entry
        $workingHour = WorkingHour::create([
            'employee_id' => $employeeId,
            'date'        => $today,
            'clock_in'    => $clockInTime,
        ]);

        return back()->with('success', _lang('Clocked in successfully'));
    }

    public function clock_out(Request $request) {
        $employeeId = auth()->user()->employee->id;
        $today      = Carbon::today();

        $workingHour = WorkingHour::where('employee_id', $employeeId)
            ->whereDate('date', $today)
            ->whereNull('clock_out')
            ->first();

        if (! $workingHour) {
            return back()->with('error', _lang('No active clock-in found'));
        }

        $clockOutTime = Carbon::now()->format('H:i:s'); // Store only time

        // Calculate worked hours
        $clockInTime        = Carbon::parse($workingHour->clock_in);
        $clockOutTimeParsed = Carbon::parse($clockOutTime);
        $workHours          = $clockInTime->diffInMinutes($clockOutTimeParsed) / 60; // Convert to hours

        $workingHour->update([
            'clock_out'   => $clockOutTime,
            'work_hour'   => number_format($workHours, 2), // Store with 2 decimal places
            'hour_deduct' => 0,
        ]);

        return back()->with('success', _lang('Clocked out successfully'));
    }
}
