<?php
namespace App\Http\Controllers;

use Exception;
use App\Models\Task;
use App\Models\User;
use App\Models\TaskLog;
use Illuminate\Http\Request;
use App\Notifications\TaskCommented;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class CommentController extends Controller {
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct() {
        date_default_timezone_set(get_timezone());

        $this->middleware(function ($request, $next) {
            if ($request->tenant->package->task_management != 1) {
                if ($request->ajax()) {
                    return response()->json(['result' => 'error', 'message' => _lang('Sorry, This module is not available in your subscription plan !')]);
                }
                return back()->with('error', _lang('Sorry, This module is not available in your subscription plan !'));
            }

            return $next($request);
        });
    }

    public function add_comment(Request $request, $tenant, $taskId) {
        $request->validate([
            'content' => 'required|string|max:1000',
        ]);

        $task    = Task::findOrFail($taskId);
        $comment = $task->comments()->create([
            'user_id' => Auth::id(),
            'content' => $request->content,
        ]);

        $notifiables = $task->users
            ->merge([$task->user])
            ->unique('id')
            ->reject(function ($user) {
                return $user->id === Auth::id();
            });

        try {
            Notification::send($notifiables, new TaskCommented($comment));
        } catch (Exception $e) {
            //Nothing
        }   

        // Log comment posting action
        TaskLog::create([
            'task_id'     => $task->id,
            'user_id'     => Auth::id(),
            'action'      => 'comment_posted',
            'description' => "Comment posted on task.",
            'data'        => ['content' => $comment->content],
        ]);

        return back()->with('success', _lang('Comment added successfully'));
    }
}
